<?php
/**
 * License Verification API
 * POST /api/verify.php
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, X-License-Key, X-Signature, X-Timestamp');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config.php';

// Get request data
$licenseKey = $_SERVER['HTTP_X_LICENSE_KEY'] ?? '';
$signature = $_SERVER['HTTP_X_SIGNATURE'] ?? '';
$timestamp = intval($_SERVER['HTTP_X_TIMESTAMP'] ?? 0);

// Also accept POST body
if (empty($licenseKey)) {
    $input = json_decode(file_get_contents('php://input'), true);
    $licenseKey = $input['license_key'] ?? '';
    $signature = $input['signature'] ?? '';
    $timestamp = intval($input['timestamp'] ?? time());
}

if (empty($licenseKey)) {
    jsonResponse(['status' => 'error', 'message' => 'License key required'], 400);
}

try {
    // Find instance by license key
    $stmt = $pdo->prepare("
        SELECT i.*, t.code as tenant_code, t.name as tenant_name, t.status as tenant_status
        FROM instances i
        JOIN tenants t ON i.tenant_id = t.id
        WHERE i.license_key = ? AND i.is_active = 1
    ");
    $stmt->execute([$licenseKey]);
    $instance = $stmt->fetch();

    if (!$instance) {
        jsonResponse([
            'status' => 'invalid',
            'message' => 'License key not found or inactive'
        ], 401);
    }

    // Check tenant status
    if ($instance['tenant_status'] === 'suspended' || $instance['tenant_status'] === 'churned') {
        jsonResponse([
            'status' => 'suspended',
            'message' => 'Account suspended. Please contact support.'
        ], 403);
    }

    // Get active subscription
    $stmt = $pdo->prepare("
        SELECT * FROM subscriptions 
        WHERE tenant_id = ? AND status = 'active' 
        AND (expires_at IS NULL OR expires_at >= CURDATE())
        ORDER BY created_at DESC LIMIT 1
    ");
    $stmt->execute([$instance['tenant_id']]);
    $subscription = $stmt->fetch();

    if (!$subscription) {
        jsonResponse([
            'status' => 'expired',
            'message' => 'No active subscription found'
        ], 403);
    }

    // Get active modules (plan + core)
    $stmt = $pdo->prepare("
        SELECT m.code, m.name, m.icon, m.category, 'active' as status, NULL as trial_expires
        FROM subscription_modules sm
        JOIN modules m ON sm.module_code = m.code
        WHERE sm.subscription_id = ? AND sm.is_active = 1
        UNION
        SELECT code, name, icon, category, 'active' as status, NULL as trial_expires 
        FROM modules WHERE is_core = 1
    ");
    $stmt->execute([$subscription['id']]);
    $activeModules = $stmt->fetchAll();
    
    // Get trial modules (not expired)
    $stmt = $pdo->prepare("
        SELECT m.code, m.name, m.icon, m.category, 'trial' as status, mt.expires_at as trial_expires
        FROM module_trials mt
        JOIN modules m ON mt.module_code = m.code
        WHERE mt.subscription_id = ? 
          AND mt.expires_at >= CURDATE()
          AND mt.starts_at <= CURDATE()
    ");
    $stmt->execute([$subscription['id']]);
    $trialModules = $stmt->fetchAll();
    
    // Merge modules (avoid duplicates - active takes precedence)
    $activeCodes = array_column($activeModules, 'code');
    foreach ($trialModules as $trial) {
        if (!in_array($trial['code'], $activeCodes)) {
            $activeModules[] = $trial;
        }
    }
    $modules = $activeModules;

    // Update last heartbeat
    $stmt = $pdo->prepare("UPDATE instances SET last_heartbeat = NOW(), server_ip = ? WHERE id = ?");
    $stmt->execute([$_SERVER['REMOTE_ADDR'] ?? '', $instance['id']]);

    // Record Telemetry (Verify Request)
    try {
        $stmt = $pdo->prepare("
            INSERT INTO telemetry (instance_id, event_type, client_ip, recorded_at)
            VALUES (?, 'verify', ?, NOW())
        ");
        $stmt->execute([
            $instance['id'],
            $_SERVER['REMOTE_ADDR'] ?? ''
        ]);
    } catch (Exception $e) {
        // Ignore telemetry errors to not block verification
    }

    // Build response
    $response = [
        'status' => 'valid',
        'tenant' => [
            'code' => $instance['tenant_code'],
            'name' => $instance['tenant_name']
        ],
        'subscription' => [
            'plan' => $subscription['plan_type'],
            'expires_at' => $subscription['expires_at'],
            'max_users' => intval($subscription['max_users']),
            'max_storage_gb' => intval($subscription['max_storage_gb'])
        ],
        'modules' => $modules,
        'issued_at' => date('c'),
        'valid_until' => date('c', time() + CACHE_TTL)
    ];

    // Sign response
    $response['signature'] = signResponse($response);

    jsonResponse($response);

} catch (Exception $e) {
    jsonResponse([
        'status' => 'error',
        'message' => 'Server error: ' . $e->getMessage()
    ], 500);
}
